//
//  TWNoContentOverlayView.m
//  Twitch
//
//  Created by Auston Stewart on 2/21/14.
//  Copyright (c) 2014 Twitch Interactive, Inc. All rights reserved.
//

#import "TWNoContentOverlayView.h"
#import "UIFont+Twitch.h"

@implementation TWNoContentOverlayView
{
	UIView *_containerView;
	UIImageView *_iconImageView;
	UILabel *_titleLabel;
	UILabel *_messageLabel;
}

#define TWNoContentOverlayViewContainerWidth 190.f

- (id)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        self.autoresizingMask = UIViewAutoresizingFlexibleWidth | UIViewAutoresizingFlexibleHeight;
		
		// Instantiate the container
		_containerView = [[UIView alloc] initWithFrame:CGRectMake(floorf((frame.size.width - TWNoContentOverlayViewContainerWidth)/2.f), 0.f, TWNoContentOverlayViewContainerWidth, 0.f)];
		_containerView.autoresizingMask = UIViewAutoresizingFlexibleTopMargin | UIViewAutoresizingFlexibleBottomMargin | UIViewAutoresizingFlexibleLeftMargin | UIViewAutoresizingFlexibleRightMargin;
		
		// Instantiate the Icon Image View
		_iconImageView = [[UIImageView alloc] initWithFrame:CGRectZero];
		_iconImageView.contentMode = UIViewContentModeCenter;
		_iconImageView.tintColor = [UIColor twitchNoContentCellIconColor];
		_iconImageView.tintAdjustmentMode = UIViewTintAdjustmentModeNormal;
		
		// Instantiate the Title Label
		_titleLabel = [[UILabel alloc] initWithFrame:CGRectZero];
		_titleLabel.font = [UIFont twitchNoContentOverlayTitleFont];
		_titleLabel.textColor = [UIColor twitchNoContentCellTextColor];
		_titleLabel.textAlignment = NSTextAlignmentCenter;
		_titleLabel.lineBreakMode = NSLineBreakByTruncatingTail;
		_titleLabel.adjustsFontSizeToFitWidth = YES;
		_titleLabel.minimumScaleFactor = .7f;
		_titleLabel.numberOfLines = 1;
		
		// Instantiate the Message Label
		_messageLabel = [[UILabel alloc] initWithFrame:CGRectZero];
		_messageLabel.font = [UIFont twitchNoContentOverlayTextFont];
		_messageLabel.textColor = [UIColor twitchNoContentCellTextColor];
		_messageLabel.textAlignment = NSTextAlignmentCenter;
		_messageLabel.lineBreakMode = NSLineBreakByTruncatingTail;
		_messageLabel.numberOfLines = 0; // Multi-line text
		
		// Assemble the view hierarchy
		[_containerView addSubview:_iconImageView];
		[_containerView addSubview:_titleLabel];
		[_containerView addSubview:_messageLabel];
		[self addSubview:_containerView];
    }
    return self;
}

- (void)layoutSubviews
{
#define TWNoContentOverlayImageMargin 36.f
#define TWNoContentOverlayTextMargin 15.f
	[super layoutSubviews];
	
	CGSize constrainedSize = CGSizeMake(TWNoContentOverlayViewContainerWidth, 400.f);
	
	// Resize and reposition icon
	[_iconImageView sizeToFit];
	_iconImageView.frame = CGRectMake(floorf((TWNoContentOverlayViewContainerWidth - _iconImageView.frame.size.width)/2.f), 0.f, _iconImageView.frame.size.width, _iconImageView.frame.size.height);
	
	// Resize and reposition the title label
	CGSize titleSize = [_titleLabel sizeThatFits:constrainedSize];
	_titleLabel.frame = CGRectMake(0.f, CGRectGetMaxY(_iconImageView.frame) + TWNoContentOverlayImageMargin, TWNoContentOverlayViewContainerWidth, titleSize.height);
	
	// Resize and reposition the message label
	CGSize messageSize = [_messageLabel sizeThatFits:constrainedSize];
	_messageLabel.frame = CGRectMake(0.f, CGRectGetMaxY(_titleLabel.frame) + TWNoContentOverlayTextMargin, TWNoContentOverlayViewContainerWidth, messageSize.height);
	
	// Resize the reposition the container view
	_containerView.frame = CGRectMake(_containerView.frame.origin.x, floorf((self.frame.size.height - CGRectGetMaxY(_messageLabel.frame))/2.f), _containerView.frame.size.width, CGRectGetMaxY(_messageLabel.frame));
	
#undef TWNoContentOverlayImageMargin
#undef TWNoContentOverlayTextMargin
}

- (void)setIcon:(UIImage *)icon
{
	_icon = icon;
	_iconImageView.image = icon;
	[self setNeedsLayout];
}

- (void)setTitle:(NSString *)title
{
	_title = title;
	_titleLabel.text = title;
	[self setNeedsLayout];
}

- (void)setMessage:(NSString *)message
{
	_message = message;
	_messageLabel.text = message;
	[self setNeedsLayout];
}

- (void)setAttributedMessage:(NSAttributedString *)attributedMessage
{
	_attributedMessage = attributedMessage;
	_messageLabel.attributedText = attributedMessage;
	[self setNeedsLayout];
}

#undef TWNoContentOverlayViewContainerWidth

@end
